/******************************************************************************
 *
 *		 I 2 C    D R I V E R
 *
 */
#include "i2c.h"

uchar state, i2cread, errorcode, bytecount, devaddr, regaddr, *dataptr;

uchar i2cStart(void);

// States
#define IDLE			0
#define START			1
#define START_ADDR		2
#define REG_ADDR		3
#define RESTART			4
#define RESTART_ADDR	5
#define READ			6
#define READ_ACK		7
#define NACK			8
#define WRITE			9
#define STOP			10

void i2cInitialise(void) {
    // Set up pins
    SDA_TRIS = 1;
    SCK_TRIS = 1;
    // Configure SPI for I2C master mode
    SSPADD = 3;                         // 1Mz
    SSPSTATbits.SMP = 1;                // Slew rate for 1MHz
    SSPCON1bits.SSPM = 0x8;             // I2C master mode
    SSPCON1bits.SSPEN = 1;              // Enable SSP

    // Configure I2C Interrupt
    IPR1bits.SSPIP = 1;                 // high priority
}

uchar i2cSetByte(uchar device, uchar address, uchar data) {
    uchar temp;

    temp = data;
    i2cread = 0;
    bytecount = 1;
    dataptr = &temp;
    devaddr = device << 1;
    regaddr = address;
    return i2cStart();
}

uchar i2cGetByte(uchar device, uchar address) {
    uchar data;

    i2cread = 1;
    bytecount = 1;
    dataptr = &data;
    devaddr = device << 1;
    regaddr = address;
    i2cStart();
    return data;
}

uchar i2cWrite(uchar device, uchar address, uchar num, uchar *data) {
    i2cread = 0;
    bytecount = num;
    dataptr = data;
    devaddr = device << 1;
    regaddr = address;
    return i2cStart();
}

uchar i2cRead(uchar device, uchar address, uchar num, uchar *data) {
    i2cread = 1;
    bytecount = num;
    dataptr = data;
    devaddr = device << 1;
    regaddr = address;
    return i2cStart();
}

uchar i2cStart() {
    PIR1bits.SSPIF = 0;                         // Clear I2C interrupt flag
    PIE1bits.SSPIE = 1;                         // Allow I2C interrupts
    state = START;
    errorcode = 0;
    SSPCON2bits.SEN = 1;                        // Send Start
    while (state != IDLE);                      // Loop here until transaction done
    PIE1bits.SSPIE = 0;                         // Disable I2C interrupts
    return errorcode;
}

void i2cStateMachine(void) {
    switch (state) {
        case START:
        { // Start has completed
            SSPBUF = devaddr;                   // load address & write bit
            state = START_ADDR;
            break;
        }
        case START_ADDR:
        { // Device address sent
            if (SSPCON2bits.ACKSTAT) {          // Error, NACK
                errorcode = ERR_ADDR;           // Flag error & abort
                state = NACK;
            } else { // Successful
                SSPBUF = regaddr;               // load register address
                state = REG_ADDR;
            }
            break;
        }
        case REG_ADDR:
        { // Register address sent
            if (SSPCON2bits.ACKSTAT) {          // Error, NACK
                errorcode = ERR_REG;            // Flag error & abort
                state = NACK;
            } else if (i2cread == 0) {          // If writing
                SSPBUF = *dataptr++;            // load data
                bytecount--;
                state = WRITE;
            } else { // If reading
                SSPCON2bits.RSEN = 1;           // Send restart
                state = RESTART;
            }
            break;
        }
        case WRITE:
        { // Data byte sent
            if (SSPCON2bits.ACKSTAT) {          // Error, NACK
                errorcode = ERR_DATA;           // Flag error & abort
                state = NACK;
            } else if (bytecount) {             // More to write?
                SSPBUF = *dataptr++;            // load data
                bytecount--;
            } else { // Done last
                SSPCON2bits.PEN = 1;            // initiate bus stop
                state = STOP;
            }
            break;
        }
        case RESTART:
        { // Restart done
            SSPBUF = devaddr | 0x01;            // load address & read bit
            state = RESTART_ADDR;
            break;
        }
        case RESTART_ADDR:
        { // Device address sent
            if (SSPCON2bits.ACKSTAT) {          // Error, NACK
                errorcode = ERR_DATA;           // Flag error & abort
                state = NACK;
            } else { // Successful
                SSPCON2bits.RCEN = 1;           // clock in data
                state = READ_ACK;
            }
            break;
        }
        case READ_ACK:
        {
            // Byte received
            *dataptr++ = SSPBUF;
            if (--bytecount) {                  // If more to send
                SSPCON2bits.ACKDT = 0;          // send ACK
                state = READ;
            } else {                            // else
                SSPCON2bits.ACKDT = 1;          // send NACK
                state = NACK;
            }
            SSPCON2bits.ACKEN = 1;
            break;
        }
        case READ:
        { // ACK sent
            SSPCON2bits.RCEN = 1;               // clock in data
            state = READ_ACK;
            break;
        }
        case NACK:
        {
            SSPCON2bits.PEN = 1;                // initiate bus stop
            state = STOP;
            break;
        }
        default:
        case STOP:
        { // Stop sent
            state = IDLE;
        }
            break;
    }
}